#include <cassert>
#include <iostream>
#include "NimGame.hpp"
#include "NimMove.hpp"
#include "NimMoveSelector.hpp"
#include "UniformIntDistribution.hpp"

int main() {
   std::cout << "*** Game of Nim ***" << std::endl;

   // read parameters of the game
   unsigned int number_of_heaps;
   std::cout << "Number of heaps: ";
   if (!(std::cin >> number_of_heaps) || number_of_heaps == 0) {
      std::cout << "Bye!" << std::endl; return 1;
   }
   unsigned int maxtake;
   std::cout << "Maximal number of sticks that can be taken in one move: ";
   if (!(std::cin >> maxtake)) {
      std::cout << "Bye!" << std::endl; return 1;
   }

   // initialize pseudo random generator
   UniformIntDistribution rand;
   // setup game
   NimGame game(number_of_heaps, maxtake);
   unsigned int minsize; unsigned int maxsize;
   if (maxtake) {
      minsize = maxtake * 2; maxsize = maxtake * 4;
   } else {
      minsize = 1; maxsize = 7;
   }
   unsigned int range = maxsize - minsize + 1;
   for (unsigned int i = 0; i < number_of_heaps; ++i) {
      game.set_heap_size(i, rand.draw(range) + minsize);
   }

   while (!game.finished()) {
      // print current state
      std::cout << "Heaps:";
      for (unsigned int i = 0; i < number_of_heaps; ++i) {
	 std::cout << " " << game.get_heap_size(i);
      }
      std::cout << std::endl;
      NimMove move;
      if (game.get_next_player() == NimGame::PLAYER1) {
	 // human player
	 do {
	    unsigned int heap_index; unsigned int count;
	    std::cout << "Your move: ";
	    if (!(std::cin >> heap_index >> count)) {
	       std::cout << "Bye!" << std::endl; return 1;
	    }
	    move = NimMove(heap_index, count);
	 } while (!game.valid_move(move));
      } else {
	 // computer
	 if (game.nim_value() == 0) {
	    // bad luck, pick a move by random
	    NimMoveSelector selector;
	    for (unsigned int i = 0; i < number_of_heaps; ++i) {
	       unsigned int max = maxtake;
	       if (max > game.get_heap_size(i)) {
		  max = game.get_heap_size(i);
	       }
	       for (unsigned int take = 1; take <= max; ++take) {
		  selector.add(NimMove(i, take));
	       }
	    }
	 } else {
	    // find a winning move
	    bool found = false;
	    NimMoveSelector selector;
	    for (unsigned int i = 0; !found && i < number_of_heaps; ++i) {
	       unsigned int max = game.get_heap_size(i);
	       if (maxtake && max > maxtake) max = maxtake;
	       for (unsigned int count = 1; count <= max; ++count) {
		  NimGame test = game;
		  NimMove testmove(i, count);
		  test.execute_move(testmove);
		  if (test.nim_value() == 0) {
		     selector.add(testmove); found = true;
		  }
	       }
	    }
	    assert(found);
	    move = selector.get();
	 }
	 std::cout << "Taking " << move.get_count() << " from heap "
	    << move.get_heap() << std::endl;
      }
      game.execute_move(move);
   }
   switch (game.winner()) {
      case NimGame::PLAYER1:
	 std::cout << "Congratulations!" << std::endl; break;
      case NimGame::PLAYER2:
	 std::cout << "You lose!" << std::endl; break;
   }
}
