#ifndef ROMBERG_HPP
#define ROMBERG_HPP

/* The following code has been copied from
   http://mymathlib.webtrellis.net/c_source/quadrature/romberg/rombergs_method.c
   (C) 2004 RLH
   and adapted to C++ by AFB
*/

#include <cmath>
#include <functional>

static const double richardson[] = {
  3.333333333333333333e-01, 6.666666666666666667e-02, 1.587301587301587302e-02,
  3.921568627450980392e-03, 9.775171065493646139e-04, 2.442002442002442002e-04,
  6.103888176768601599e-05, 1.525902189669642176e-05, 3.814711817595739730e-06,
  9.536752259018191355e-07, 2.384186359449949133e-07, 5.960464832810451556e-08,
  1.490116141589226448e-08, 3.725290312339701922e-09, 9.313225754828402544e-10,
  2.328306437080797376e-10, 5.820766091685553902e-11, 1.455191522857861004e-11,
  3.637978807104947841e-12, 9.094947017737554185e-13, 2.273736754432837583e-13,
  5.684341886081124604e-14, 1.421085471520220567e-14, 3.552713678800513551e-15,
  8.881784197001260212e-16, 2.220446049250313574e-16
};

template<typename Real = double>
class Romberg {
   public:
      static constexpr unsigned int max_cols = 1 +
	 sizeof(richardson)/sizeof(richardson[0]);

      Romberg(std::function<Real(Real)> f, Real a, Real b, Real tolerance) :
	    integral(0), ok(false) {
	 Real h = b - a;
	 Real dt[max_cols];
	 Real integralsum = 0;

	 // Initialize first column, dt[0], to the numerical estimate
	 // of the integral using the trapezoidal rule with a step size of h.
	 dt[0] = 0.5 * h * (f(a) + f(b));

	 // For each possible succeeding column, halve the step size,
	 // calculate the composite trapezoidal rule using the new step
	 // size, and up date preceeding columns using Richardson
	 // extrapolation.

	 for (int k = 1; k < max_cols; k++) {
	    Real old_h = h;

	    // Calculate T(f,h/2,a,b) using T(f,h,a,b)
	    h *= 0.5;
	    integralsum = 0.0;
	    for (Real x = a + h; x < b; x += old_h) {
	       integralsum += f(x);
	    }
	    integralsum = h * integralsum + 0.5 * dt[0];

	    // Calculate the Richardson Extrapolation to the limit
	    Real delta;
	    for (int j = 0; j < k; j++) {
	       delta = integralsum - dt[j];
	       dt[j] = integralsum;
	       integralsum += richardson[j] * delta;
	    }

	    // If the magnitude of the change in the extrapolated estimate
	    // for the integral is less than the preassigned tolerance,
	    // we are finished

	    if (std::fabs(delta) < tolerance) {
	       ok = true;
	       break;
	    }

            // Store the current estimate in the kth column.
	    dt[k] = integralsum;
	 }
	 integral = integralsum;
      }

   bool converged() const {
      return ok;
   }

   operator Real() const {
      return integral;
   }

   private:
      Real integral;
      bool ok;
};

#endif
