#ifndef ROOTCNT_HPP
#define ROOTCNT_HPP

#include <cfloat>
#include <cmath>
#include <functional>
#include <limits>
#include "romberg.hpp" // used for numerical integration

/* numerical computation of the first derivative,
   using the five point method */

template<typename F, typename Real = double>
class NumF1 {
   public:
      NumF1(F& f) : f(f) {
      }
      Real operator()(Real x) const {
	 // out of Numerical Recipes by William H. Press
	 auto epsilon = std::numeric_limits<Real>::epsilon();
	 Real h = std::sqrt(epsilon) * x;
	 volatile Real temp = x + h; // suppress optimizations
	 h = temp - x;
	 // five point method
	 return (-f(x + 2*h) + 8*f(x + h) - 8*f(x - h) + f(x - 2*h))/12/h;
      }
   private:
      F& f;
};

/* numerical computation of the second derivative,
   using the five point method */

template<typename F, typename Real = double>
class NumF2 {
   public:
      NumF2(F& f) : f(f) {
      }
      Real operator()(Real x) const {
	 auto epsilon = std::numeric_limits<Real>::epsilon();
	 Real h = std::sqrt(epsilon) * x;
	 volatile Real temp = x + h; // suppress optimizations
	 h = temp - x;
	 // five point method for second derivative
	 return (-f(x + 2*h) + 16*f(x + h) - 30*f(x)
		     + 16*f(x - h) - f(x - 2*h))/12/h/h;
      }
   private:
      F& f;
};

/*
   Compute number of simple roots of a function f: [a,b] --> |R and a,b in |R
   where
     (1) f is twice continuously differentiable in [a,b]
     (2) f(a) * f(b) != 0

   The computation is done using a Kronecker-Picard integral,
   see V.P. Plagianakos et al: "Locating and computing in parallel
   all the simple roots of special functions using PVM",
   Journal of Computational and Applied Mathematics 133 (2001) 545-554

   Template parameters:
      F     function f
      F1    function f'
      F2    function f''
      Real  type which must support atan, fabs, and regular arithmetic ops
            (by default double)

   Examples:

      // specify first and second derivative explicitly
      unsigned int count1 = get_root_count(a, b,
	 [](double x) { return sin(x); },    // f
	 [](double x) { return cos(x); },    // f'
	 [](double x) { return -sin(x); });  // f''

      // let f' and f'' be computed numerically
      unsigned int count2 = get_root_count(a, b, j0);
*/

template<typename F, typename Real = double>
unsigned int get_root_count(Real a, Real b, F f) {
   NumF1<F, Real> f1(f);
   NumF2<F, Real> f2(f);
   return get_root_count(a, b, f, f1, f2);
}
template<typename F, typename F1, typename F2, typename Real = double>
unsigned int get_root_count(Real a, Real b, F f, F1 f1, F2 f2) {
   Romberg<Real> integral([&](Real x) {
      Real fx = f(x);
      Real f1x = f1(x);
      Real f1xsq = f1x * f1x;
      return (fx * f2(x) - f1xsq) / (fx * fx + f1xsq);
   }, a, b, 1E-2);
   constexpr auto PI = std::acos(Real(-1.0));
   Real n = - (integral - std::atan(f1(b)/f(b)) +
		  std::atan(f1(a)/f(a))) / PI;
   return (unsigned int) (n + 0.1);
}

#endif
